// HDLM 8 API
// Version 3.0.8
// License: GNU Lesser General Public License (LGPL), see http://www.gnu.org/licenses/lgpl-3.0.de.html
// supported OSes:
// - Linux
// - Android
// - OS X
// - iOS
// - Windows (using MinGW)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <assert.h>
#ifdef __MINGW32__
#include <winsock2.h>
#else
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#endif
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include "utf/utf.h"
#include "hdlm_api.h"

#define HDLM_NET_PORT	"24871"
#define	NET_MODE_UNIX	"<mode>unix</mode>\n"
#define SMALL_BUF_SIZE	(64)
#define	READ_BUF_SIZE			(98304)
#define READ_BUF_RESIDUAL_SIZE	(READ_BUF_SIZE/2)
#define COMMAND_ARG_SIZE		(4096)

static void *command_arg=0;
static char *netbuf=0;
static char *netbuf_residual=0;
static int netbuf_residual_size=0;

typedef void (*command_dispatch_t)(char *c);
	
struct listener {
	listener_t 		work;
	void			*user;
	struct listener *next;
	int				align_dummy;
};

struct dispatch {
	command_dispatch_t work;
	struct listener *listeners;
};

static struct dispatch command_dispatch[COMMAND_LAST+1];

#ifndef strtok_r
char *strtok_r(char *str, const char *delim, char **save) {
	
    char *res, *last;

    if( !save )
        return strtok(str, delim);
    if( !str && !(str = *save) )
        return NULL;
    last = str + strlen(str);
    if( (*save = res = strtok(str, delim)) )
    {
        *save += strlen(res);
        if( *save < last )
            (*save)++;
        else
            *save = NULL;
    }
    return res;
}
#endif

#ifndef strsep
char *strsep(char **stringp, const char *delim) {
	
	char *s;
	const char *spanp;
	int c, sc;
	char *tok;

	if ((s = *stringp) == NULL)
		return (NULL);
	for (tok = s;;) {
		c = *s++;
		spanp = delim;
		do {
			if ((sc = *spanp++) == c) {
				if (c == 0)
					s = NULL;
				else
					s[-1] = 0;
				*stringp = s;
				return (tok);
			}
		} while (sc != 0);
	}
	/* NOTREACHED */
}
#endif

#ifndef strlcpy
size_t strlcpy(char* dst, const char* src, size_t siz) {

	// original BSD code
	char *d = dst;
    const char *s = src;
    size_t n = siz;
 
     /* Copy as many bytes as will fit */
     if (n != 0 && --n != 0) {
             do {
                     if ((*d++ = *s++) == 0)
                             break;
             } while (--n != 0);
     }

     /* Not enough room in dst, add NUL and traverse rest of src */
     if (n == 0) {
             if (siz != 0)
                     *d = '\0';              /* NUL-terminate dst */
             while (*s++)
                     ;
     }
	return(s - src - 1);    /* count does not include NUL */
}
#endif

static int connect_sock (const char *host, const char *service, const char *transport) {

	struct hostent      *phe;   // host information
	struct servent      *pse;   // service information
	struct sockaddr_in sin;     // internet ep address         
	int s;		                // socket descriptor
	int type;					// socket type    

	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;

	// map service name to port number
	if ((pse = getservbyname(service, transport))) {
		sin.sin_port = pse->s_port;
	}
	else if ((sin.sin_port = htons((unsigned short)atoi(service))) == 0) {
		err("can't get \"%s\" service entry\n", service);
		return -1;
	}
	    

	if ((phe = gethostbyname(host))) {
		memcpy(&sin.sin_addr, phe->h_addr, phe->h_length);
	}
	else if ((sin.sin_addr.s_addr = inet_addr(host)) == INADDR_NONE) {
		err("can't get \"%s\" host entry\n", host);
		return -1;
	}
	    

#ifndef __ANDROID__
	struct protoent     *ppe;   // pointer to protocol information entry
	// map transport protocol name to protocol number
	if ((ppe = getprotobyname(transport)) == 0) {
		err("can't get \"%s\" protocol entry\n", transport);
		return -1;
	}
#endif

	if (strcmp(transport, "udp") == 0)
	    type = SOCK_DGRAM;
	else
	    type = SOCK_STREAM;

#ifdef __ANDROID__
	s = socket(PF_INET, type, 0);
#else
	s = socket(PF_INET, type, ppe ? ppe->p_proto : 0);
#endif

	if (s < 0) {
		err("can't create socket: %s\n", strerror(errno));
		return -1;
	}

	// connect the socket
	if (connect(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		close(s);
		s = -1;
	}
	
	return s;
}

static ssize_t net_write(int handle, const void *buf, size_t nbyte) {

	ssize_t wc=0;
	ssize_t wl=0;
	
	while (nbyte) {
#ifndef __MINGW32__		
		wc = write(handle, buf, nbyte);
#else
		wc = send(handle, buf, nbyte, 0);
#endif		
		wl += wc;
		if ((wc == nbyte) || (wc <= 0))
			break;
			
		buf += wc;
		nbyte -= wc;
	}
	return wl;
}

static ssize_t net_read(int handle, void *buf, size_t nbyte) {
#ifndef __MINGW32__
		return read(handle, buf, nbyte);
#else
		return recv(handle, buf, nbyte, 0);
#endif				
}

static inline float scan_float(char *s) {

	char *word, *brkt;
	float ret;
	
	word = strtok_r(s, ".", &brkt); 
	ret = (float)atoi(word);

	word = strtok_r(NULL, ".", &brkt);

	if (word)
		ret += (float)atoi(word) * ((*s=='-') ? -0.1 : 0.1);

	return ret;
}

static inline void scan_meter(char *s, int *id, float *peak_db, float *gr_db) {
	// fast replacement for sscanf(s, "%*d,%d,%f,%f", &arg->out_id, &arg->peak_db, &arg->gr_db);
	char *ap;

	strsep(&s, ",");
	
	ap = strsep(&s, ",");
	*id = atoi(ap);
	
	ap = strsep(&s, ",");
 	*peak_db = /*(float)atoi*/scan_float(ap);

	ap = strsep(&s, ",");
	*gr_db = /*(float)atoi*/scan_float(ap);
}

static inline void scan_integer_array(char *s, int *array, int count, int *conn_id) {

	char *ap;

	strsep(&s, ",");
	
	ap = strsep(&s, ",");
	*conn_id = atoi(ap);
	
	while (*ap && count--) {
		ap = strsep(&s, ",");
	 	*(array++) = atoi(ap);
	}
}

static inline void scan_float_array(char *s, float *array, int count, int *ch_id) {

	char *ap;

	strsep(&s, ",");
	strsep(&s, ",");
	
	ap = strsep(&s, ",");
	*ch_id = atoi(ap);
	
	while (*ap && count--) {
		ap = strsep(&s, ",");
	 	*(array++) = atof/*scan_float*/(ap);
	}
}

static inline void scan_command_id(char *s, int *command_id/*, int *conn_id*/) {
	// fast replacement for sscanf(c, "%d,%d", &command_id, &cid);
	char *ap;
	char tmp[8];
	strlcpy(tmp, s, sizeof(tmp));
	char *c = tmp;

	ap = strsep(&c, ",");
	*command_id = atoi(ap);
}

static void dispatch_COMMAND_OUT_METER(char *c) {

	struct arg_out_meter *arg = command_arg;
	//sscanf(c, "%*d,%d,%f,%f", &arg->out_id, &arg->peak_db, &arg->gr_db);
	scan_meter(c, &arg->out_id, &arg->peak_db, &arg->gr_db);
	arg->conn_id=0;
	//printf("out: %d, peak: %1.1f, gr: %1.1f\n", arg->out_id, arg->peak_db, arg->gr_db);
} 

static void dispatch_COMMAND_BUS_METER(char *c) {

	struct arg_bus_meter *arg = command_arg;
	//sscanf(c, "%*d,%d,%f,%f", &arg->bus_id, &arg->peak_db, &arg->gr_db);
	scan_meter(c, &arg->bus_id, &arg->peak_db, &arg->gr_db);
	arg->conn_id=0;		
	//printf("bus: %d, peak: %1.1f, gr: %1.1f\n", arg->bus_id, arg->peak_db, arg->gr_db);
}

static void dispatch_COMMAND_SYS_INPUTSTATE(char *c) {

	struct arg_sys_inputstate *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->analog_overflow, &arg->digital_state);
	arg->conn_id=0;		
}

static void dispatch_COMMAND_BUS_LABEL(char *c) {

	struct arg_bus_label *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%"SYS_BUS_LABEL_SIZE_S"s", &arg->conn_id, &arg->bus_id, arg->label);
	if (*arg->label!=',') {
		//printf("%s\n", arg->label);
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_BUS_MUTE(char *c) {

	struct arg_bus_mute *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->bus_id, &arg->mute);
}

static void dispatch_COMMAND_BUS_GAIN(char *c) {

	struct arg_bus_gain *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->gain);
}

static void dispatch_COMMAND_BUS_PHASE(char *c) {

	struct arg_bus_phase *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->bus_id, &arg->phase);
}

static void dispatch_COMMAND_BUS_COMP_THRES(char *c) {

	struct arg_bus_comp_thres *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->thres);
}

static void dispatch_COMMAND_BUS_COMP_RATIO(char *c) {

	struct arg_bus_comp_ratio *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->ratio);
}

static void dispatch_COMMAND_BUS_COMP_KNEE(char *c) {

	struct arg_bus_comp_knee *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->knee);
}

static void dispatch_COMMAND_BUS_PEAK_OVERSHOOT(char *c) {

	struct arg_bus_peak_overshoot *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->overshoot);
}

static void dispatch_COMMAND_BUS_COMP_ATTACK(char *c) {

	struct arg_bus_comp_attack *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->attack);
}

static void dispatch_COMMAND_BUS_COMP_RELEASE(char *c) {

	struct arg_bus_comp_release *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->release);
}

static void dispatch_COMMAND_BUS_PEAK_RELEASE(char *c) {

	struct arg_bus_peak_release *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->bus_id, &arg->peakrelease);
}

static void dispatch_COMMAND_BUS_GROUP_LOCAL(char *c) {

	struct arg_bus_group_local *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->bus_id, &arg->group_local);
}

static void dispatch_COMMAND_BUS_GROUP_NET(char *c) {

	struct arg_bus_group_net *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->bus_id, &arg->group_net);
}

static void dispatch_COMMAND_BUS_PATCH(char *c) {

	struct arg_bus_patch *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->bus_id, &arg->patch);
}

static void dispatch_COMMAND_OUT_GROUPING(char *c) {
	//*32,<from_connid>,<group_lsp ch0>,...,<group_lsp ch7>,<lsp id ch0>,...,<lsp id ch7>			
	struct arg_out_grouping *arg = command_arg;
	scan_integer_array(c, arg->group_lsp, SYS_OUT_COUNT*2, &arg->conn_id);
	//int  i; for (i=0; i< SYS_OUT_COUNT; i++)	printf("group_lsp %d: id_lsp %d\n", arg->group_lsp[i], arg->id_lsp[i]);
}

static void dispatch_COMMAND_OUT_GROUP_CONFIG(char *c) {
	//*33,<from_connid>,<group_lsp>,<outmode ch0>,...,<outmode ch7>		
	struct arg_out_group_config *arg = command_arg;
	scan_integer_array(c, &arg->group_lsp, SYS_OUT_COUNT+1, &arg->conn_id);
	//int  i; for (i=0; i< SYS_OUT_COUNT; i++)	printf("ch: %d, group_lsp %d: outmode %d\n", i, arg->group_lsp, arg->outmode[i]);
}

static void dispatch_COMMAND_OUT_GROUP_LSP_LABEL(char *c) {

	struct arg_out_group_lsp_label *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%"SYS_LSP_LABEL_SIZE_S"s", &arg->conn_id, &arg->group_lsp, arg->label);
	if (*arg->label!=',') {
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_OUT_GROUP_LABEL(char *c) {

	struct arg_out_group_label *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%"SYS_OUT_LABEL_SIZE_S"s", &arg->conn_id, &arg->group_lsp, arg->label);
	if (*arg->label!=',') {
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_OUT_GROUP_GAIN(char *c) {

	struct arg_out_group_gain *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->group_lsp, &arg->gain);
}

static void dispatch_COMMAND_OUT_GROUP_PHASE(char *c) {

	struct arg_out_group_phase *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->group_lsp, &arg->phase);
}

static void dispatch_COMMAND_OUT_GROUP_GRP_LOCAL(char *c) {

	struct arg_out_group_grp_local *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->group_lsp, &arg->group_local);
}

static void dispatch_COMMAND_OUT_GROUP_GRP_NET(char *c) {

	struct arg_out_group_grp_net *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->group_lsp, &arg->group_net);
}

static void dispatch_COMMAND_OUT_GROUP_INPUT(char *c) {

	struct arg_out_group_input *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->group_lsp, &arg->input_bus);
}

static void dispatch_COMMAND_OUT_GROUP_DELAY(char *c) {

	struct arg_out_group_delay *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->group_lsp, &arg->delay);
}

static void dispatch_COMMAND_OUT_GROUP_DELAY_MS(char *c) {

	struct arg_out_group_delay_ms *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f", &arg->conn_id, &arg->group_lsp, &arg->delay_ms);
}

static void dispatch_COMMAND_OUT_MUTE(char *c) {
	struct arg_out_mute	*arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->out_id, &arg->mute);
}

static void dispatch_COMMAND_LSP_FILE(char *c) {
	//	64,<from_connid>,<lspid>,<id_parent>,<version>,<is_folder>,<is_locked>,<ways>,<label>
	struct arg_lsp_file *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%d,%d,%d,%d,%d,%"SYS_LSP_LABEL_SIZE_S"s", &arg->conn_id, &arg->lsp_id, &arg->id_parent, &arg->is_used, &arg->is_folder, &arg->is_locked, &arg->ways, arg->label);
	if (*arg->label!=',') {
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_LSP_FILE_START(char *c) {
	//	65,<from_connid>,<partial>
	struct arg_lsp_file_start *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->partial);
}

static void dispatch_COMMAND_LSP_FILE_END(char *c) {
	//	66,<from_connid>
	struct arg_lsp_file_end *arg = command_arg;
	sscanf(c, "%*d,%d", &arg->conn_id);
}

static void dispatch_COMMAND_SYS_NETWORK(char *c) {
	//	21,<from_connid>,<0=reserved>,<0=reserved>,<ip-0>,<ip-1>,<ip-2>,<ip-3>,<netmask-bits>,<gateway-0>,<gateway-1>,<gateway-2>,<gateway-3>
	struct arg_sys_network *arg = command_arg;
	unsigned int dhcp_ip_hex;
	int mode;
	sscanf(c, "%*d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d", &arg->conn_id, &dhcp_ip_hex, &mode, &arg->ip[0], &arg->ip[1], &arg->ip[2], &arg->ip[3], &arg->netmask_bits, &arg->gateway[0], &arg->gateway[1], &arg->gateway[2], &arg->gateway[3]);
	arg->mode = mode & 0xff;
	arg->wifi_on = !!(mode & (1<<8));
	arg->dhcp_ip[3] = dhcp_ip_hex & 0xff;
	arg->dhcp_ip[2] = (dhcp_ip_hex >> 8) & 0xff;
	arg->dhcp_ip[1] = (dhcp_ip_hex >> 16) & 0xff;
	arg->dhcp_ip[0] = (dhcp_ip_hex >> 24) & 0xff;
}			

static void dispatch_COMMAND_SYS_SERVICEMODE(char *c) {
	//	20,<from_connid>,<1=fixed-on>
	struct arg_sys_servicemode *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->on);
}			

static void dispatch_COMMAND_SYS_DEVICENAME(char *c) {
	//	22,<from_connid>,<name>
	struct arg_sys_devicename *arg = command_arg;
	arg->device_name[0]=0;
	sscanf(c, "%*d,%d,%"SYS_MISC_DEVICENAME_SIZE_S"s", &arg->conn_id, arg->device_name);
	if (*arg->device_name!=',') {
		urlunesc(arg->device_name);
		strcpy(arg->device_name_upper, arg->device_name);
		utfstrtoupper(arg->device_name_upper);
	}
	else {
		*arg->device_name=0;
		*arg->device_name_upper=0;
	}
}

static void dispatch_COMMAND_SYS_WLAN_PASSWORD(char *c) {
	// 112,<from_connid>,<password>
	struct arg_sys_wlan_password *arg = command_arg;
	arg->wlan_password[0]=0;
	sscanf(c, "%*d,%d,%"SYS_NET_WLAN_PASSWORD_SIZE_S"s", &arg->conn_id, arg->wlan_password);
	if (*arg->wlan_password!=',') {
		urlunesc(arg->wlan_password);
	}
	else {
		*arg->wlan_password=0;
	}
}

static void dispatch_COMMAND_SYS_DELAYUNIT(char *c) {
	//	23,<from_connid>,<0=meter|1=feet|2=milliseconds>,<temperature K>,<*max value>
	struct arg_sys_delayunit *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f,%f", &arg->conn_id, &arg->delay_unit, &arg->delay_temp, &arg->delay_max);
}			

static void dispatch_COMMAND_SYS_AMPLIFIERS(char *c) {
	//	24,<from_connid>,<outid>,<gain>,<peak voltage>			
	struct arg_sys_amplifiers *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%d", &arg->conn_id, &arg->outid, &arg->gain, &arg->peak_voltage);
}

static void dispatch_COMMAND_SYS_SECURITY(char *c) {
	//  27,<from_connid>,<flags SYS_SECURITY_xxx or -1 for info>
	struct arg_sys_security *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->flags);
}

static void dispatch_COMMAND_SYS_DIGITALMODE(char *c) {
	//  63,<from_connid>,<SYS_MISC_DIGITAL_MODE_xxx>
	struct arg_sys_digitalmode *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->flags);
}

static void dispatch_COMMAND_SYS_FLAGS(char *c) {
	//  62,<from_connid>,<SYS_MISC_FLAGS_xxx>
	struct arg_sys_flags *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->flags);
}

static void dispatch_COMMAND_PRJ_FILE(char *c) {
	//	96,<from_connid>,<project_id>,<is_locked>,<label>
	struct arg_prj_file *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%d,%"SYS_PROJECTNAME_SIZE_S"s", &arg->conn_id, &arg->project_id, &arg->is_locked, arg->label);
	if (*arg->label!=',') {
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_PRJ_FILE_START(char *c) {
	//	97,<from_connid>,<partial>
	struct arg_prj_file_start *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->partial);
}

static void dispatch_COMMAND_PRJ_FILE_END(char *c) {
	//	98,<from_connid>
	struct arg_prj_file_end *arg = command_arg;
	sscanf(c, "%*d,%d", &arg->conn_id);
}

static void dispatch_COMMAND_PRJ_USB_FILE(char *c) {
	//*109,<from_connid>,<file_id>,<reserved>,<label>
	struct arg_prj_usb_file *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%d,%"SYS_PROJECTNAME_SIZE_S"s", &arg->conn_id, &arg->file_id, &arg->reserved, arg->label);
	if (*arg->label!=',') {
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_PRJ_USB_FILE_START(char *c) {
	//*110,<from_connid>,<partial>
	struct arg_prj_usb_file_start *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->partial);
}

static void dispatch_COMMAND_PRJ_USB_FILE_END(char *c) {
	//*111,<from_connid>
	struct arg_prj_usb_file_end *arg = command_arg;
	sscanf(c, "%*d,%d", &arg->conn_id);
}

static void dispatch_COMMAND_PRJ_USB_EXPORT_ACK(char *c) {
	//*107,<from_connid>,<project_id>		
	struct arg_prj_usb_export_ack *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->project_id);
}

static void dispatch_COMMAND_PRJ_USB_IMPORT_ACK(char *c) {
	//*108,<from_connid>,<file_id>			
	struct arg_prj_usb_import_ack *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->file_id);
}

static void dispatch_COMMAND_PRJ_STAT(char *c) {
	// 100,<from_connid>,<project_id>,<is_locked>,<*saved_is_locked|°0>,<*is_dirty|°0>,<label>
	struct arg_prj_stat *arg = command_arg;
	arg->label[0]=0;
	sscanf(c, "%*d,%d,%d,%d,%d,%d,%"SYS_PROJECTNAME_SIZE_S"s", &arg->conn_id, &arg->project_id, &arg->is_locked, &arg->save_is_locked, &arg->is_dirty, arg->label);
	if (*arg->label!=',') {
		urlunesc(arg->label);
		strcpy(arg->label_upper, arg->label);
		utfstrtoupper(arg->label_upper);
	}
	else {
		*arg->label=0;
		*arg->label_upper=0;
	}
}

static void dispatch_COMMAND_PRJ_FILES_DIRTY(char *c) {
	//*105,<from_connid> [the project db has changed. clients have to refresh via INFO_PRJ_TREE]
	struct arg_prj_files_dirty *arg = command_arg;
	//printf("files dirty\n");
	sscanf(c, "%*d,%d", &arg->conn_id);
}

static void dispatch_COMMAND_SYS_MESSAGE(char *c) {
	//	25,<from_connid>,<type 0=info|1=warning|2=error>,<message>
	struct arg_sys_message *arg = command_arg;
	arg->msg[0]=0;
	sscanf(c, "%*d,%d,%d,%"SYS_MESSAGE_SIZE_S"s", &arg->conn_id, &arg->type, arg->msg);
	if (*arg->msg!=',') {
		urlunesc(arg->msg);
	}
	else {
		*arg->msg=0;
	}
}

static void dispatch_COMMAND_SYS_FWUPDATE(char *c) {
	//  26,<from_connid>,<key>
	struct arg_sys_fwupdate *arg = command_arg;
	sscanf(c, "%*d,%d,%d", &arg->conn_id, &arg->key);
}

static void dispatch_COMMAND_BUS_EQ_CURVE(char *c) {
	// *13,0,<busid>,<mag dB 0>,<mag dB 1>,...,<mag dB CURVES_WIDTH-1>
	struct arg_eq_curve *arg = command_arg;
	arg->conn_id=0;
	scan_float_array(c, arg->curve, DSP_CURVES_WIDTH, &arg->id);
}

static void dispatch_COMMAND_OUT_GROUP_EQ_CURVE(char *c) {
	// *48,0,<group_lsp>,<mag dB 0>,<mag dB 1>,...,<mag dB CURVES_WIDTH-1>
	struct arg_eq_curve *arg = command_arg;
	arg->conn_id=0;
	scan_float_array(c, arg->curve, DSP_CURVES_WIDTH, &arg->id);
}

static void dispatch_COMMAND_BUS_EQ_FLAGS(char *c) {
	//	14,<from_connid>,<busid>,<flags BUS_EQ_FLAGS_xxx>
	struct arg_eq_flags *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->id, &arg->flags);
}

static void dispatch_COMMAND_OUT_GROUP_EQ_FLAGS(char *c) {
	//	49,<from_connid>,<group_lsp>,<flags OUT_GROUP_EQ_FLAGS_xxx>
	struct arg_eq_flags *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d", &arg->conn_id, &arg->id, &arg->flags);
}

static void dispatch_COMMAND_BUS_GEQ_SET(char *c) {
	//	12,<from_connid>,<busid>,<geqid[0..30]>,<gain dB>
	struct arg_geq_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%f", &arg->conn_id, &arg->id, &arg->eq_id, &arg->gain);
}

static void dispatch_COMMAND_BUS_PEQ_SET(char *c) {
	//	16,<from_connid>,<busid>,<peqid[0..SYS_BUS_PEQ_COUNT-1]>,<freq>,<q>,<gain dB>,<bypass>
	struct arg_peq_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%f,%f,%f,%d", &arg->conn_id, &arg->id, &arg->eq_id, &arg->freq, &arg->q, &arg->gain, &arg->bypass);
}

static void dispatch_COMMAND_OUT_GROUP_PEQ_SET(char *c) {
	//	45,<from_connid>,<group_lsp>,<peqid[0..SYS_OUT_PEQ_COUNT-1]>,<freq>,<q>,<gain dB>,<bypass>
	struct arg_peq_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%f,%f,%f,%d", &arg->conn_id, &arg->id, &arg->eq_id, &arg->freq, &arg->q, &arg->gain, &arg->bypass);
}

static void dispatch_COMMAND_BUS_SHELFEQ_SET(char *c) {
	//	17,<from_connid>,<busid>,<shelfeqid[0..SYS_BUS_SHELFEQ_COUNT-1]>,<freq>,<gain dB>,<bypass>
	struct arg_shelfeq_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%f,%f,%d", &arg->conn_id, &arg->id, &arg->eq_id, &arg->freq, &arg->gain, &arg->bypass);
}

static void dispatch_COMMAND_OUT_GROUP_SHELFEQ_SET(char *c) {
	//	46,<from_connid>,<group_lsp>,<shelfeqid[0..SYS_OUT_SHELFEQ_COUNT-1]>,<freq>,<gain dB>,<bypass>
	struct arg_shelfeq_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%f,%f,%d", &arg->conn_id, &arg->id, &arg->eq_id, &arg->freq, &arg->gain, &arg->bypass);
}

static void dispatch_COMMAND_BUS_HPF_SET(char *c) {
	//	18,<from_connid>,<busid>,<freq>,<type HPF_TYPE_xxx>,<bypass>
	struct arg_hpf_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f,%d,%d", &arg->conn_id, &arg->id, &arg->freq, &arg->type, &arg->bypass);
}

static void dispatch_COMMAND_OUT_GROUP_HPF_SET(char *c) {
	//	47,<from_connid>,<group_lsp>,<freq>,<type HPF_TYPE_xxx>,<bypass>
	struct arg_hpf_set *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%f,%d,%d", &arg->conn_id, &arg->id, &arg->freq, &arg->type, &arg->bypass);
}

static void dispatch_COMMAND_SYS_INFO(char *c) {
	//	122,<from_connid>,<model>,<hardware_version>,<fpga_dsp_version>,<fpga_io_version>,<server_version>,<ethmacmsb>,<ethmaclsb>,<linux_version_string> [all version numbers are hex: 00.00 e.g. 0200 for 2.0 or 022a for 2.2a BUT linux_version_string is a string; ethmac is hex]
	struct arg_sys_info *arg = command_arg;
	sscanf(c, "%*d,%d,%d,%d,%d,%d,%x,%x,%x,%127s", &arg->conn_id, &arg->model, &arg->hardware_version, &arg->fpga_dsp_version, &arg->fpga_io_version, &arg->server_version, &arg->ethmacmsb, &arg->ethmaclsb, arg->linux_version_string);
}

static void work(char *c, const int conn_id) {

	// scan command, find listeners and fire them
	int command_id;//, cid;
	scan_command_id(c, &command_id/*, &cid*/);

	if ((command_id >= 0) && (command_id <= COMMAND_LAST) && command_dispatch[command_id].listeners && command_dispatch[command_id].work) {
		command_dispatch[command_id].work(c); // fire dispatch worker
		
		struct listener **listener = &command_dispatch[command_id].listeners;

			while (*listener) {
				//printf("listener %lx with %s\n", *listener, c);
				(*listener)->work(command_arg, (*listener)->user); // fire listener
				listener = &(*listener)->next;
			}
	}
}

static void queue_work(char *queue, const int conn_id) {
	// find all info commands with \n separator and work them step by step
	char *c = queue;
	char *next = queue;
	for(;;) {
		if (*c == 0) break;
		if (*c == '\n') {
			*c = 0;
			work(next, conn_id);
			next = c+1;
		}
		c++;
	}
}

void hdlm_api_exit(void) {

	if (command_arg)
		free(command_arg);
	
	if (netbuf)
		free(netbuf);

	if (netbuf_residual)
		free(netbuf_residual);
	
#ifdef __MINGW32__
	WSACleanup();
#endif
}

int hdlm_api_init(void) {
	
#ifdef __MINGW32__
	WSADATA wsa;
	if (WSAStartup(MAKEWORD(2,2),&wsa) != 0) {
        err("hdlm_api_init: WSAStartup() (%d)", WSAGetLastError());
        return -1;
    }
#endif
	
	memset(&command_dispatch, 0, sizeof(command_dispatch));
	command_dispatch[COMMAND_OUT_METER].work = 				dispatch_COMMAND_OUT_METER;
	command_dispatch[COMMAND_BUS_METER].work = 				dispatch_COMMAND_BUS_METER;
	command_dispatch[COMMAND_BUS_LABEL].work = 				dispatch_COMMAND_BUS_LABEL;
	command_dispatch[COMMAND_BUS_MUTE].work = 				dispatch_COMMAND_BUS_MUTE;
	command_dispatch[COMMAND_BUS_GAIN].work = 				dispatch_COMMAND_BUS_GAIN;
	command_dispatch[COMMAND_BUS_PHASE].work = 				dispatch_COMMAND_BUS_PHASE;
	command_dispatch[COMMAND_BUS_COMP_THRES].work = 		dispatch_COMMAND_BUS_COMP_THRES;
	command_dispatch[COMMAND_BUS_COMP_RATIO].work = 		dispatch_COMMAND_BUS_COMP_RATIO;
	command_dispatch[COMMAND_BUS_COMP_KNEE].work = 			dispatch_COMMAND_BUS_COMP_KNEE;
	command_dispatch[COMMAND_BUS_PEAK_OVERSHOOT].work =		dispatch_COMMAND_BUS_PEAK_OVERSHOOT;
	command_dispatch[COMMAND_BUS_COMP_ATTACK].work =		dispatch_COMMAND_BUS_COMP_ATTACK;
	command_dispatch[COMMAND_BUS_COMP_RELEASE].work =		dispatch_COMMAND_BUS_COMP_RELEASE;
	command_dispatch[COMMAND_BUS_PEAK_RELEASE].work =		dispatch_COMMAND_BUS_PEAK_RELEASE;
	command_dispatch[COMMAND_BUS_GROUP_LOCAL].work = 		dispatch_COMMAND_BUS_GROUP_LOCAL;
	command_dispatch[COMMAND_BUS_GROUP_NET].work = 			dispatch_COMMAND_BUS_GROUP_NET;
	command_dispatch[COMMAND_BUS_PATCH].work = 				dispatch_COMMAND_BUS_PATCH;
	command_dispatch[COMMAND_BUS_EQ_CURVE].work = 			dispatch_COMMAND_BUS_EQ_CURVE;
	command_dispatch[COMMAND_BUS_GEQ_SET].work = 			dispatch_COMMAND_BUS_GEQ_SET;
	command_dispatch[COMMAND_BUS_PEQ_SET].work = 			dispatch_COMMAND_BUS_PEQ_SET;
	command_dispatch[COMMAND_BUS_SHELFEQ_SET].work = 		dispatch_COMMAND_BUS_SHELFEQ_SET;
	command_dispatch[COMMAND_BUS_HPF_SET].work = 			dispatch_COMMAND_BUS_HPF_SET;
	command_dispatch[COMMAND_BUS_EQ_FLAGS].work = 			dispatch_COMMAND_BUS_EQ_FLAGS;
	command_dispatch[COMMAND_OUT_GROUPING].work = 			dispatch_COMMAND_OUT_GROUPING;
	command_dispatch[COMMAND_OUT_GROUP_CONFIG].work = 		dispatch_COMMAND_OUT_GROUP_CONFIG;
	command_dispatch[COMMAND_OUT_GROUP_LSP_LABEL].work =	dispatch_COMMAND_OUT_GROUP_LSP_LABEL;
	command_dispatch[COMMAND_OUT_GROUP_LABEL].work = 		dispatch_COMMAND_OUT_GROUP_LABEL;
	command_dispatch[COMMAND_OUT_GROUP_GAIN].work = 		dispatch_COMMAND_OUT_GROUP_GAIN;
	command_dispatch[COMMAND_OUT_GROUP_PHASE].work = 		dispatch_COMMAND_OUT_GROUP_PHASE;
	command_dispatch[COMMAND_OUT_GROUP_GRP_LOCAL].work =	dispatch_COMMAND_OUT_GROUP_GRP_LOCAL;
	command_dispatch[COMMAND_OUT_GROUP_GRP_NET].work = 		dispatch_COMMAND_OUT_GROUP_GRP_NET;
	command_dispatch[COMMAND_OUT_GROUP_INPUT].work = 		dispatch_COMMAND_OUT_GROUP_INPUT;
	command_dispatch[COMMAND_OUT_GROUP_DELAY].work = 		dispatch_COMMAND_OUT_GROUP_DELAY;
	command_dispatch[COMMAND_OUT_GROUP_DELAY_MS].work = 	dispatch_COMMAND_OUT_GROUP_DELAY_MS;
	command_dispatch[COMMAND_OUT_MUTE].work = 				dispatch_COMMAND_OUT_MUTE;
	command_dispatch[COMMAND_OUT_GROUP_PEQ_SET].work = 		dispatch_COMMAND_OUT_GROUP_PEQ_SET;
	command_dispatch[COMMAND_OUT_GROUP_SHELFEQ_SET].work =	dispatch_COMMAND_OUT_GROUP_SHELFEQ_SET;
	command_dispatch[COMMAND_OUT_GROUP_HPF_SET].work = 		dispatch_COMMAND_OUT_GROUP_HPF_SET;
	command_dispatch[COMMAND_OUT_GROUP_EQ_FLAGS].work = 	dispatch_COMMAND_OUT_GROUP_EQ_FLAGS;
	command_dispatch[COMMAND_OUT_GROUP_EQ_CURVE].work = 	dispatch_COMMAND_OUT_GROUP_EQ_CURVE;
	command_dispatch[COMMAND_LSP_FILE].work = 				dispatch_COMMAND_LSP_FILE;
	command_dispatch[COMMAND_LSP_FILE_START].work = 		dispatch_COMMAND_LSP_FILE_START;
	command_dispatch[COMMAND_LSP_FILE_END].work = 			dispatch_COMMAND_LSP_FILE_END;
	command_dispatch[COMMAND_SYS_NETWORK].work = 			dispatch_COMMAND_SYS_NETWORK;
	command_dispatch[COMMAND_SYS_SERVICEMODE].work = 		dispatch_COMMAND_SYS_SERVICEMODE;
	command_dispatch[COMMAND_SYS_DEVICENAME].work = 		dispatch_COMMAND_SYS_DEVICENAME;
	command_dispatch[COMMAND_SYS_DELAYUNIT].work = 			dispatch_COMMAND_SYS_DELAYUNIT;
	command_dispatch[COMMAND_SYS_AMPLIFIERS].work = 		dispatch_COMMAND_SYS_AMPLIFIERS;
	command_dispatch[COMMAND_SYS_SECURITY].work = 			dispatch_COMMAND_SYS_SECURITY;
	command_dispatch[COMMAND_SYS_DIGITALMODE].work = 		dispatch_COMMAND_SYS_DIGITALMODE;
	command_dispatch[COMMAND_SYS_FLAGS].work = 				dispatch_COMMAND_SYS_FLAGS;
	command_dispatch[COMMAND_SYS_INPUTSTATE].work = 		dispatch_COMMAND_SYS_INPUTSTATE;
	command_dispatch[COMMAND_PRJ_FILE].work = 				dispatch_COMMAND_PRJ_FILE;
	command_dispatch[COMMAND_PRJ_FILE_START].work = 		dispatch_COMMAND_PRJ_FILE_START;
	command_dispatch[COMMAND_PRJ_FILE_END].work = 			dispatch_COMMAND_PRJ_FILE_END;
	command_dispatch[COMMAND_PRJ_STAT].work = 				dispatch_COMMAND_PRJ_STAT;
	command_dispatch[COMMAND_PRJ_FILES_DIRTY].work = 		dispatch_COMMAND_PRJ_FILES_DIRTY;
	command_dispatch[COMMAND_PRJ_USB_EXPORT_ACK].work = 	dispatch_COMMAND_PRJ_USB_EXPORT_ACK;
	command_dispatch[COMMAND_PRJ_USB_IMPORT_ACK].work = 	dispatch_COMMAND_PRJ_USB_IMPORT_ACK;
	command_dispatch[COMMAND_PRJ_USB_FILE].work = 			dispatch_COMMAND_PRJ_USB_FILE;
	command_dispatch[COMMAND_PRJ_USB_FILE_START].work = 	dispatch_COMMAND_PRJ_USB_FILE_START;
	command_dispatch[COMMAND_PRJ_USB_FILE_END].work = 		dispatch_COMMAND_PRJ_USB_FILE_END;
	command_dispatch[COMMAND_SYS_MESSAGE].work = 			dispatch_COMMAND_SYS_MESSAGE;
	command_dispatch[COMMAND_SYS_FWUPDATE].work = 			dispatch_COMMAND_SYS_FWUPDATE;
	command_dispatch[COMMAND_SYS_WLAN_PASSWORD].work = 		dispatch_COMMAND_SYS_WLAN_PASSWORD;
	command_dispatch[COMMAND_SYS_INFO].work = 				dispatch_COMMAND_SYS_INFO;

	netbuf = malloc(READ_BUF_SIZE);
	if (netbuf==NULL) {
		perror("hdlm_api_init: netbuf==NULL");
		return -1;
	}

	netbuf_residual = malloc(READ_BUF_RESIDUAL_SIZE);
	if (netbuf_residual==NULL) {
		perror("hdlm_api_init: netbuf_residual==NULL");
		return -1;
	}
	netbuf_residual_size = 0;
	
	command_arg = malloc(COMMAND_ARG_SIZE);
	if (command_arg==NULL) {
		perror("hdlm_api_init: command_arg==NULL");
		return -1;
	}
	
	return 0;
}

void hdlm_api_add_listener(const int command_id, listener_t work_listener, void *user) {

	assert((command_id >= 0) && (command_id <= COMMAND_LAST));
	assert(command_dispatch[command_id].work != NULL);
	assert(work_listener != NULL);
	struct listener **listener = &command_dispatch[command_id].listeners;
	struct listener **last = NULL; 
	while (*listener) {
		last = listener;
		listener = &(*listener)->next;
	}
	*listener = calloc(1, sizeof(struct listener));
	if (last && *last)
		(*last)->next = *listener;
	(*listener)->work = work_listener;
	(*listener)->user = user;
}

int hdlm_api_read_nonblock(int net_handle, const int conn_id) {

	int n;
	int ret = 0;
	struct timeval timeout;
	static fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(net_handle, &rfds);

	timeout.tv_sec = 0;
	timeout.tv_usec = 100;

	if (select(net_handle+1, &rfds, NULL, NULL, &timeout)) {
				
		if (netbuf_residual_size > 0)
			memcpy(netbuf, netbuf_residual, netbuf_residual_size);
		
		n = net_read(net_handle, netbuf+netbuf_residual_size, READ_BUF_SIZE-netbuf_residual_size-1);

		if (n > 0) {

			n += netbuf_residual_size;
			netbuf_residual_size = 0;
			
			if (*(netbuf+n-1) == '\n') {
				*(netbuf+n) = 0; // terminate
			}
			else {
			
				// split in respect to network packet size
				char *s=netbuf+n;
				int l=0;
			
				while ((l<READ_BUF_RESIDUAL_SIZE) && (s >= netbuf) && (*(--s) != '\n')) l++;
				memcpy(netbuf_residual, s+1, l);
				*(netbuf_residual+l) = 0;
				*(s+1) = 0;
				netbuf_residual_size = l;
				n -= l;
			}

			if (n >= READ_BUF_SIZE) 
				err("!!! network_handle: n >= READ_BUF_SIZE");

			queue_work(netbuf, conn_id);
			//printf("%s\n\n", netbuf);
		}
		else {
			err("network_handle: n <= 0 (aborting)");
			close(net_handle);
			ret = -1; // connection closed
		}	
	}

	return ret;
}

// connects to HDLM using IP address
// returns network file handle on success
// conn_id returns the connection ID and is optional (may be NULL)
int hdlm_api_connect(const char *ip_addr, int *conn_id) {

	int net_handle;
	char *buf[SMALL_BUF_SIZE];
		
	log("hdlm_api_connect: trying...");
	net_handle = connect_sock(ip_addr, HDLM_NET_PORT, "tcp");

	if (net_handle >= 0) {
	
		log("hdlm_api_connect: connected.");

		// establish connection
		net_write(net_handle, NET_MODE_UNIX, sizeof(NET_MODE_UNIX));

		net_read(net_handle, (void*)buf, SMALL_BUF_SIZE);

		if (conn_id) {
			
			sscanf((char*)buf, "%d", conn_id);
		}
		
		netbuf_residual_size = 0;
	}
	
	return net_handle;
}

// disconnect from HDLM
void hdlm_api_disconnect(int net_handle) {

	close(net_handle);
}

int hdlm_api_write(int handle, const char *buf) {
	
	return (int) net_write(handle, (const void*)buf, strlen(buf));
}
