// Minimal API example for Seeburg HDLM 8
// call this program with the HDLM's IP to get its project list
// call this program with the HDLM's IP and a project id to load a new project
// tested operating systems:
// - Linux (gcc)
// - OS X (gcc)
// - Windows (using MinGW and 'make -f Makefile.windows' command)

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "hdlm_api.h"

#ifdef __MINGW32__
// prevent usleep() warning
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#endif

static int prj_list_complete=0;
static int connection_ready=0;

static void listener_prj_file(void *par, void *user) {

	const struct arg_prj_file *arg = par;

	printf("id=%d name='%s'\n", arg->project_id, arg->label);
}

static void listener_prj_file_start(void *par, void *user) {

	printf("--- Project File Start\n");
}

static void listener_prj_file_end(void *par, void *user) {

	printf("--- Project File End\n");
	prj_list_complete = 1;
}

static void listener_meter_out(void *par, void *user) {

	connection_ready = 1; // when metering starts we can be sure that all system information was transmitted and we may send new commands to the HDLM
}

int main (int argc, char* argv[]) {
	
	int handle;
	int conn_id;
	int stat;
	int load_id=-1;
	char cmd_buf[512];
	
	if (argc < 2) {
		printf("usage: minimal <hdlm-ip> [project-id to load]\n");
		return 0;
	}
	
	if (argc == 3) {
		load_id = atoi(argv[2]);
	}
	
	hdlm_api_init(); 
	
	if (load_id == -1) {

		hdlm_api_add_listener(COMMAND_PRJ_FILE, listener_prj_file, NULL);	
		hdlm_api_add_listener(COMMAND_PRJ_FILE_START, listener_prj_file_start, NULL);	
		hdlm_api_add_listener(COMMAND_PRJ_FILE_END, listener_prj_file_end, NULL);			
	}

	hdlm_api_add_listener(COMMAND_OUT_METER, listener_meter_out, NULL);			
	
	handle = hdlm_api_connect(argv[1], &conn_id);
	
	if (handle < 0) {

		err("connection error.");
		return -1;
	}

	
	for (;;) {

		stat = hdlm_api_read_nonblock(handle, conn_id);
	
		if (stat < 0) // error
			break;
	
		if (prj_list_complete) // project list completed, end program
			break;
	
		if ((load_id > 0) && connection_ready) {
			
			// load the project id invoked by program argument list			
			sprintf(cmd_buf, "%d,%d,%d\n", COMMAND_PRJ_LOAD, conn_id, load_id); // COMMAND_PRJ_LOAD			(99)	// °99,<from_connid>,<project_id>
			hdlm_api_write(handle, cmd_buf);
			printf("Project loaded\n");
			prj_list_complete=1; // we must come at least one time through the loop until it will be savely transmitted on Wind*ws
		}
	
		usleep(1000);
	}
	
	hdlm_api_disconnect(handle);
	hdlm_api_exit();
	return 0;
}

